#!/bin/perl
#********************************************************************************
# IBM Spectrum Protect Plus
#
# name: vsnapperf
#
# desc: Simulates typical IBM Spectrum Protect Plus workload patterns for assessing
#       the suitability of disk storage.  The workloads are
#       1) sequential write simulating backup ingest
#
# usage:  perl vsnapperf.pl -help 
#
# Notice: This program is provided as a tool intended for use by IBM Internal,
#         IBM Business Partners, and IBM Customers. This program is provided as is,
#         without support, and without warranty of any kind expressed or implied.
#
# (C) Copyright International Business Machines Corp. 2013, 2021
#********************************************************************************

#use warnings;

use Cwd 'abs_path';
use Getopt::Long;
use Time::HiRes qw(time sleep);

my $version = "Program version 1.4c";
my $fileStr = "SPPperfTest";

if (@ARGV < 1) {
  print "Syntax error\n";
  printSyntax();
  exit 1;
}

my $action = "";
if ($ARGV[0] eq "-help") {
  printSyntax();
  exit;
}

if($ARGV[0] eq "seqwrite" || $ARGV[0] eq "SEQWRITE" ||
   $ARGV[0] eq "seqread" || $ARGV[0] eq "SEQREAD" ||
   $ARGV[0] eq "cleanup" || $ARGV[0] eq "CLEANUP"){
  $action = $ARGV[0];
}
else {
  print "ERROR: Unknown workload type: $ARGV[0]\n";
  exit 1;
}

# Process options and defaults

my %opts = ();
$opts{fileSize} = '';
$opts{iterations} = '';
$opts{skipNum} = '';
$opts{overwrite} = '';
$opts{cleanup} = '';
$opts{numVol} = '';
$opts{volName} = '';
$opts{volSync} = '';
$opts{bSize} = '';
$opts{retention} = '';
$opts{snapshots} = '';
$opts{random} = '';

GetOptions (
  \%opts,
  'fileSize=s',
  'iterations=i',
  'retention=i',
  'skipNum=i',
  'overwrite=s',
  'cleanup=s',
  'numVol=i',
  'volName=s',
  'volSync=s',
  'bSize=s',
  'snapshots=s',
  'random=s',
) or printSyntax();

if($opts{fileSize} eq ""){
    $opts{fileSize} = "10g";
}
elsif ($opts{fileSize} !~ m/\d+[k|m|g]+/i) {
  print "ERROR: invalid -fileSize specified (expected:  numberK|M|G)\n\n";
  exit 1;
}
if($opts{numVol} eq ""){
    $opts{numVol} = 5;
}
if($opts{volName} eq ""){
   $opts{volName}="SPPvSnapPerf";
}
elsif ($opts{volName} eq "fs" || $opts{volName} eq "FS" || $opts{volName} eq "fn" || $opts{volName} eq "FN") {
  print "ERROR: The volName $opts{volName} is restricted and cannot be used\n\n";
  exit 1;
}
if($opts{volSync} eq "" || $opts{volSync} =~ m/no/i){
  $opts{volSync}="disabled";
}
elsif ($opts{volSync} =~ m/yes/i) {
  $opts{volSync}="always";
}
else {
  print "ERROR: invalid -volSync specified (expected: yes|no)\n\n";
  exit 1;
}
if($opts{bSize} eq ""){
  $opts{bSize} = "128k";
}
elsif ($opts{bSize} !~ m/\d+[k|m|g]+/i) {
  print "ERROR: invalid -bSize specified (expected:  numberK|M|G)\n\n";
  exit 1;
}
if($opts{iterations} eq ""){
  $opts{iterations} = 1;
}
if($opts{skipNum} eq ""){
  $opts{skipNum} = 0;
}
elsif ($opts{skipNum} >= $opts{iterations}) {
  print "ERROR: The value for the -iterations parameter must be greater than the -skipNum parameter\n\n";
  exit 1;
}
if($opts{overwrite} eq ""){
  $opts{overwrite} = "no";
}
elsif ($opts{overwrite} !~ m/(yes)|(no)/i) {
  print "ERROR: invalid -overwrite specified (expected: yes|no)\n\n";
  exit 1;
}
if($opts{cleanup} eq "") {
  #default case
  if ($opts{iterations} > 1)  {
    $opts{cleanup} = "no";
  }
  else {
     $opts{cleanup} = "yes";   
  }
}
elsif  ($opts{cleanup} !~ m/(yes)|(no)/i){
  print "ERROR: invalid -cleanup specified (expected: yes|no)\n\n";
  exit 1;
}
if($opts{snapshots} eq "" || $opts{snapshots} =~ m/yes/i ){
  $opts{snapshots} = "yes";
}
elsif ($opts{snapshots} =~ m/no/i) {
  $opts{snapshots} = "no";
}
else {
  print "ERROR: invalid -snapshots specified (expected: yes|no)\n\n";
  exit 1;
}
if($opts{retention} eq ""){
  $opts{retention} = $opts{iterations};
}
if($opts{random} eq "" || $opts{random} =~ m/high/){
  $opts{random} = "/dev/randhigh";
}
elsif($opts{random} =~ m/medium/){
  $opts{random} = "/dev/randmed";
}
elsif($opts{random} =~ m/low/){
  $opts{random} = "/dev/randlow";
}
else {
  print "ERROR: invalid -random specified (expected: high|medium|low)\n\n";
  exit 1;
}

# Validate the ldeedee tool can be located
my $ddName = "ldeedee";
if (-f "./".$ddName)
{
  $ddName = "./".$ddName;
}
else  # build a path based on the full path to vsnapperf.pl
{
  my $fullPath = $0;
  $fullPath =~ m/(.*)(\/|\\)(\w|\.)+$/;
  $ddName = $1.$2.$ddName;
  if (! -f $ddName)
  {
    print "ERROR: Unable to locate required file $ddName\n";
    exit (1);
  }
}
if (! -x $ddName)
{
  print "ERROR: Unable to execute required file $ddName (check permissions)\n";
  exit (1);
}

my @createdVols = qw();
@vol = SetupZFS($action);
@fsList=();
$i = 0;
foreach $fs(@vol){
   if($fs ne ""){
     $fsList[$i] = "\/vsnap\/" . $fs;
     $i++;
   } 
}

my $out = "";
my $outputFile = "";
my $summaryLog = "vsnapperf_summary.txt";

$out = localtime();
lprint ($summaryLog, $out);
dprint ($summaryLog, "\n===================================================================\n");
dprint ($summaryLog, ": IBM Spectrum Protect Plus vSnap storage performance test  ($version)\n");
dprint ($summaryLog, ":\n");
dprint ($summaryLog, ": Running with action $action\n");
dprint ($summaryLog, ":\n");
dprint ($summaryLog, ": Number of iterations:\t$opts{iterations}\n: Skipping iterations:\t$opts{skipNum}\n: Number of volumes:\t$opts{numVol}\n");
dprint ($summaryLog, ":\n");
dprint ($summaryLog, ": Block size:\t$opts{bSize}\n: File size:\t$opts{fileSize}\n: Syncwrite:\t$opts{volSync}\n: Snapshots:\t$opts{snapshots}\n");
dprint ($summaryLog, ": Cleanup:\t$opts{cleanup}\n: Random:\t$opts{random}\n");
dprint ($summaryLog, ": Volumes:\t@fsList\n");
dprint ($summaryLog, ":\n");
dprint ($summaryLog, "===================================================================\n");

# Main loop processing all of the iterations

my $minTput = 9999999999;
my $maxTput = 0;
my $totAvgTput = 0;
my $grandTotGB;

if ($action =~ m/cleanup/i) {
  if ($#createdVols >= 0) {
    $outputFile = "vsnapperfcleanup.txt";
    runCleanup(1, @createdVols);
  }
  else {
    print "ERROR: There are no matching volumes to cleanup.  Check volName and numVol options\n";
    exit 1;
  }
}
else {
  print ": Beginning I/O test.\n";
  for($k=$opts{skipNum}+1;$k < $opts{iterations}+1;$k++){
    RunTest($action,$fileStr,$k);

    # If retention option is being used, cleanup snapshots in excess of retention
    if($k > ($opts{retention})){
       $snap_del=$k-$opts{retention};
       $snap_name = "DedupSnap" . $snap_del;
       for($l=0;$l < $opts{numVol};$l++){
          $cmd = "sudo zfs destroy $vol[$l]\@$snap_name";
          system($cmd);
       }
    } 
  }
  $totAvgTput = $totAvgTput / ($opts{iterations} - $opts{skipNum});

  if ($opts{iterations} > 1) {
    dprint ($summaryLog,": All iterations complete!\n");
    dprint ($summaryLog,":\n");
    $out = sprintf (": Total processed    (GiB):\t%.1f\n", $grandTotGB);
    dprint ($summaryLog, $out);
    $out = sprintf (": Minimum throughput (MiB/sec):\t%.2f\n", $minTput);
    dprint ($summaryLog, $out);
    $out = sprintf (": Maximum throughput (MiB/sec):\t%.2f\n", $maxTput);
    dprint ($summaryLog, $out);
    $out = sprintf (": Average throughput (MiB/sec):\t%.2f\n", $totAvgTput);
    dprint ($summaryLog, $out);
  }

  # end of main routine
}

# Cleanup if required
if ($action !~ m/cleanup/i && $opts{cleanup} =~ m/yes/i) {
  runCleanup(0, @createdVols);
}

dprint ($summaryLog, "===================================================================\n");

sub SetupZFS($)
{
   my $actionType = shift(@_);

   #find out the vsnap filesystem
   $vsnapVol = "";
   $output = `df|grep vpool 2>&1`;
   if($output =~/^vpool(\d+).*/)
   {
       $vsnapVol = "vpool" . $1;
   }
   else{
      print "ERROR: There is no vSnap vpool mounted on the system\n";
      exit 1;
   }

   @volList = undef;
   for($i=0;$i<$opts{numVol};$i++){
     my $volNum = $i+1;
     $volList[$i] = $vsnapVol . "\/" . $opts{volName} . $volNum;
     my $create = 1;
     $rc = system("sudo df|grep $volList[$i] 2>&1 >/dev/null");
     if(!$rc){      # Handle file system which already exists
       if ($actionType =~ m/cleanup/i) {
         push (@createdVols, $volList[$i]);
       }
       $create = 0;
       my $next = $opts{skipNum} + 1;
       my $last = $opts{iterations} - 1;
       if (($actionType !~ m/cleanup/i) && (-e "/vsnap/".$volList[$i]."/".$fileStr.$next || -e "/vsnap/".$volList[$i]."/".$fileStr.$last)) {
         if ($opts{overwrite} eq "no" && $action !~ m/seqread/i) {
           print "ERROR: Files already exist in /vsnap/$volList[$i]. Re-run with -skipNum or -overwrite=yes\n";
           exit 1;
         }
       }
     }

     if ($create == 1 && ($actionType !~ m/cleanup/i)) {
       $rc =  system("sudo zfs create $volList[$i]");
       if($rc != 0){
          print "ERROR: Failure creating vSnap volume: $volList[$i]\n";
          exit 1;
       }
       # Record the created volume for future cleanup
       push (@createdVols, $volList[$i]);

       $rc = system("sudo zfs set sync=$opts{volSync} $volList[$i]");
       if($rc !=0){
          print "ERROR: Failure setting syncwrite attribute for volume: $volList[$i]\n";
          exit 1;
       }

     }
   }
   return @volList;
}

sub RunTest($$$)
{
  my $actionType = shift(@_);
  my $fileName = shift(@_);
  my $num = shift(@_);
  my $createForRead = 0;

  $fileName = $fileName . $num%$opts{retention};
  $outputFile = "vsnapperf" . $num . ".txt";
  if(-e ".\/$outputFile"){
     #print "$outputFile already exists. Removing it for a new test run\n";
     system("sudo rm -f $outputFile");
  }
  print (": Run $num. Output: $outputFile Status: RUNNING  [             ]\b\b\b\b\b\b\b\b\b\b\b\b\b");
  lprint ($summaryLog, ": Run $num. Output: $outputFile Status: ");
  my $localTime = localtime();
  lprint ($outputFile, $localTime."\n");
  lprint ($outputFile, "********Start $ddName process**********\n");

  # convert file size to same unit as block size and determine block count
  my $blockBytes = 0;
  $opts{bSize} =~ m/(\d+)(k|m|g)/i;
  if ($2 eq "k" || $2 eq "K")
  {
    $blockBytes = int ($1*1024);
  }
  elsif ($2 eq "m" || $2 eq "M")
  {
    $blockBytes = int ($1*1024*1024);
  }
  elsif ($2 eq "g" || $2 eq "G")
  {
    $blockBytes = int ($1*1024*1024*1024);
  }
  else
  {
    $blockBytes = $1;
  }

  my $fileBytes = 0;
  $opts{fileSize} =~ m/(\d+)(k|m|g)/i;
  if ($2 eq "k" || $2 eq "K")
  {
    $fileBytes = int ($1*1024);
  }
  elsif ($2 eq "m" || $2 eq "M")
  {
    $fileBytes = int ($1*1024*1024);
  }
  elsif ($2 eq "g" || $2 eq "G")
  {
    $fileBytes = int ($1*1024*1024*1024);
  }

  my $numBlocks = 0;
  if ($fileBytes < $blockBytes)
  {
    $blockBytes = $fileBytes;
    $numBlocks = 1;
  }
  else
  {
    $numBlocks = int ($fileBytes / $blockBytes);
  }

  # Create files for seqread if needed
  if ($actionType =~ m/seqread/i) {
    for($j=0; $j < $opts{numVol}; $j++) { 
      if ( (-s $fsList[$j]."/".$fileName) < $fileBytes ) {
        $createForRead = 1;
      }
    }
  }

  my $totalBytes = $fileBytes * $opts{numVol};
  my $totalSeconds = 0;
  my $startTime = 0;
  my $endTime = 0;
  my $totalIOs = 0;

  # Handle seqwrite action or creating file for seqread
  if ($actionType =~ m/seqwrite/i || $createForRead == 1) {
    $totalIOs = 0;
    $startTime = time();
    for($j=0; $j < $opts{numVol}; $j++) {
      my $cmd = "sudo $ddName if=$opts{random} bs=$opts{bSize} of=$fsList[$j]\/$fileName count=$numBlocks  >> $outputFile 2>&1 &";
      lprint ($outputFile, "Execute Command: $cmd\n");
      system($cmd);
      $totalIOs += $numBlocks;
    }
    sleep 2;
    waitForProcs("ldeedee");
    $endTime = time();
  }

  # Handle seqread action
  if ($actionType =~ m/seqread/i) {
    $startTime = time();
    for($j=0; $j < $opts{numVol}; $j++) {
      my $cmd = "sudo $ddName if=$fsList[$j]\/$fileName bs=$opts{bSize} of=/dev/null count=$numBlocks  >> $outputFile 2>&1 &";
      lprint ($outputFile, "Execute Command: $cmd\n");
      system($cmd);
      $totalIOs += $numBlocks;
    }
    sleep 2;
    waitForProcs("ldeedee");
    $endTime = time();
  }

  print ("\b\b\b\b\b\b\b\b\b\b\bCOMPLETE   ");
  lprint ($summaryLog, "COMPLETE   ");
  lprint ($outputFile, "********End $ddName**********\n");
  $localTime = localtime();
  lprint ($outputFile, $localTime."\n");

  # calculate elapsed time
  $elapsedTime = $endTime - $startTime;
  my $totMB = $totalBytes / 1024 / 1024;
  my $totGB = $totalBytes / 1024 / 1024 / 1024;
  my $avgTput = $totMB / $elapsedTime;
  my $avgIOPS = $totalIOs / $elapsedTime;
 
  # Capture totals across multiple runs
  if ($avgTput < $minTput) {
    $minTput = $avgTput;
  }
  if ($avgTput > $maxTput) {
    $maxTput = $avgTput;
  }
  $totAvgTput += $avgTput;
  $grandTotGB += $totGB;

  lprint ($outputFile, "Total size (GiB): $totGB\n");
  lprint ($outputFile, "Elapsed time (seconds): $elapsedTime\n");
  lprint ($outputFile, "Throughput (MiB/sec): $avgTput\n");
  lprint ($outputFile, "IOPS: $avgIOPS\n");

  if ($opts{iterations} > 1) {
    $out = sprintf ("Seconds: %.2f  Total GiB: %.1f  MiB/sec: %.2f  IOPS: %.2f\n", $elapsedTime, $totGB, $avgTput, $avgIOPS);
    dprint ($summaryLog, $out);
  }
  else {
    dprint ($summaryLog,"                       \n");
    dprint ($summaryLog,":\n");
    dprint ($summaryLog,"===================================================================\n");
    dprint ($summaryLog,": RESULTS\n");
    if ($action =~ m/write/i) {
      $out = sprintf (": Total written (GiB):\t%.2f\n",$totGB);
      dprint ($summaryLog, $out);
    }
    elsif ($action =~ m/read/i) {
      $out = sprintf (": Total read (GiB):\t%.2f\n",$totGB);
      dprint ($summaryLog, $out);
    }
    $out = sprintf (": Throughput (MiB/sec):\t%.2f\n",$avgTput);
    dprint ($summaryLog, $out);
    $out = sprintf (": Average IOPS:\t\t%.2f\n",$avgIOPS);
    dprint ($summaryLog, $out);
    $out = sprintf (": Elapsed Time (sec):\t%.2f\n",$elapsedTime);
    dprint ($summaryLog, $out);
  }

  # Create snapshots of all volumes unless -snapshots=no
  if ($opts{snapshots} eq "yes")
  {
    lprint ($outputFile, "\n********Start snapshot creation**********\n");
    my $snapName = "DedupSnap" . $num;
    $startTime = time();
    for($i=0;$i<$opts{numVol};$i++){
       my $cmd = "sudo zfs snapshot $vol[$i]\@$snapName >> $outputFile 2>&1";
       `$cmd`;
       lprint ($outputFile, "Created snapshot $vol[$i]\@$snapName\n");
    }
    waitForProcs("zfs snapshot");
    $endTime = time();

    lprint ($outputFile, "********End snapshot creation**********\n");
    $localTime = localtime();
    lprint ($outputFile, $localTime."\n");
    $elapsedTime = $endTime - $startTime;
    lprint ($outputFile, "Elapsed time (seconds): $elapsedTime\n");
    sleep(2);
  }

  # Collect some other useful statistics for this run
  lprint ($outputFile, "\n********Collect vSnap pool statistics**********\n");
  system("sudo vsnap pool show >> $outputFile 2>&1");
  system("sudo zpool status -D  >> $outputFile 2>&1");
  system("sudo vsnap system stats >> $outputFile 2>&1");
  system("sudo arc_summary >> $outputFile 2>&1");
  system("sudo cat /proc/spl/kstat/zfs/arcstats >> $outputFile 2>&1");
  system("sudo free -m >> $outputFile 2>&1");
  system("sudo cat /proc/meminfo >> $outputFile 2>&1");
  system("sudo ps aux >> $outputFile 2>&1");

}

sub runCleanup($@)
{
  my $prompt = shift (@_);
  my @cleanVols = @_;
  my $userinput = "";
  my $v;

  lprint ($outputFile, "\nPerforming volume cleanup\n");
  if ($prompt) {
    print ": WARNING: You are about to delete the following volumes, all files contained within them,\n";
    print ": and all associated snapshots.\n";
    print ":\n";
    lprint ($outputFile, "\nThe user has elected to cleanup the following volumes\n");
    foreach $v (@cleanVols) {
      lprint ($outputFile, "$v\n");
      print ":   $v\n";
    }
    print ":\n";  
    print ": To continue, enter 'YES' in uppercase, or 'quit'\n";
    $userinput="NO";
    while (($userinput ne "YES") && ($userinput !~ m/quit/i))
    {
      $userinput = "";
      print ": Continue? : ";
      $userinput = <STDIN>;
      chomp($userinput);
    }
    if ($userinput eq "quit")
    {
      print ": Quitting ...\n";
      exit (1);
    }
  }
  print ":\n";
  print ": Cleaning up volumes.  Please be patient, this can take a while ...\n";
  foreach $v (@cleanVols) {
    if ($v =~ m/fs\d+/i || $v =~ m/fs\d+/i) {
      print "ERROR: Volume $v appears to be a vSnap production volume, skipping cleanup\n";
    }
    else {
      print ": Cleaning volume $v\n";
      $cmd = "sudo zfs destroy -r $v";
      lprint ($outputFile, ": Running command $cmd\n");
      system($cmd);
    }
  }
}

sub waitForProcs($)
{
  my $procName = shift(@_);
  my $dotCount = 0;
  my $sleepCount = 0;
  my $check = 1;
  while($check){
     my  $ps_output = `sudo ps -aef|grep \"$procName\"`;
     my @ldeedee_process = split("\n", $ps_output);
     $ps_output = @ldeedee_process;
     if($ps_output <= 2){
        $check = 0;
        for (my $x=0; $x < ($dotCount % 10); $x++)
        {
          print "\b";
        }
     }
     elsif ($procName =~ m/ldeedee/) {
        sleep(0.1);
        $sleepCount++;
        if (!($sleepCount % 10)) {
          print ".";
          $dotCount++;
          if (!($dotCount % 10)) {
            print ".\b\b\b\b\b\b\b\b\b\b\b";
          }
        }
     }
  }
}


sub lprint ($$)
{
  if (@_ != 2)
  {
    print "USAGE: lprint logfile string\n";
    return 1;
  }
  $file = shift(@_);
  $string = shift(@_);

  # It's possible that the log is not inited
  if ($file ne "")
  {
     # Attempt to open the log file for appending
     if (! open (LOG, ">>$file"))
     {
       print "lprint: could not write to log file: $file!\n";
       print "$string";
       return 1;
     }
  }
  # Append the string to the log
  print LOG "$string";

  # Close the log file
  close LOG;

  return 0;
}

sub dprint ($$)       #  Print to the log file and stdout
{
  if (@_ != 2)
  {
    print "USAGE: dprint logfile string\n";
    return 1;
  }
  $file = shift(@_);
  $string = shift(@_);

  # It's possible that the log is not inited
  if ($file ne "")
  {
     # Attempt to open the log file for appending
     if (! open (LOG, ">>$file"))
     {
       print "dprint: could not write to log file: $file!\n";
       print "$string";
       return 1;
     }
  }
  # Append the string to the log
  print LOG "$string";

  # Print to stdout
  print $string;

  # Close the log file
  close LOG;

  return 0;
}

sub dprintf ($$$)       #  Print to the log file and stdout
{
  if (@_ != 3)
  {
    print "USAGE: dprint logfile string\n";
    return 1;
  }
  $file = shift(@_);
  $string = shift(@_);

  # It's possible that the log is not inited
  if ($file ne "")
  {
     # Attempt to open the log file for appending
     if (! open (LOG, ">>$file"))
     {
       print "dprint: could not write to log file: $file!\n";
       print "$string";
       return 1;
     }
  }
  # Append the string to the log
  print LOG "$string";

  # Print to stdout
  print $string;

  # Close the log file
  close LOG;

  return 0;
}

sub printSyntax
{
  print "\n  vsnapperf - IBM Spectrum Protect Plus vSnap performance testing tool ($version)\n";
  print "\n  Command: $0 <workload> [options]\n";
  print "  Workloads:\n";
  print "    seqwrite     Simulates streaming backup to vSnap with seq writes\n";
  print "    seqread      Simulates streaming restore from vSnap with seq reads\n";
  print "                 Will use existing files if they exist and are large enough,\n";
  print "                 otherwise they will be created.\n";
  print "    cleanup      Deletes volumes, files, and snapshots from previous tests after prompting\n";
  print "  Options:\n";
  print "    -fileSize:   Size of file to create under each volume in K,M, or G.  Example: 1g (default: 10g)\n";
  print "    -numVol:     Number of volume to run the test (default 5)\n";
  print "    -volName:    Name prefix of volumes (default SPPvSnapPerf)\n";
  print "    -iterations: Specifies the number of repetitions using different files each time (default 1)\n";
  print "    -skipNum:    Skip the first specified number of iterations (default 0)\n";
  print "    -overwrite:  Allow the test to overwrite existing filesons yes|no (default no)\n";
  print "    -cleanup:    Automatically cleanup volumes, files, and snapshots that are created during the\n";
  print "                 test yes|no (default yes with single iteration, no with multiple iterations)\n";
  print "    -snapshots:  Take snapshots at the end of each iteraction yes|no (default yes)\n\n";
  print "    -retention:  Specifiy how many snapshots to keep before deleting oldest snapshot each\n";
  print "                 iteration (default = infinite)\n";
  print "    -random:     Control pseudo-random data generator with values of low|medium|high (default: high)\n";
  print "    -bSize:      Specifes the block size used for read/write (default 128K)\n";
  print "    -volSync:    Force synchronous writes with values of yes|no (default no)\n";
  print "  Examples:\n";
  print "    perl vsnapperf.pl seqwrite -fileSize 100g\n";
  print "    perl vsnapperf.pl seqwrite -fileSize 50g -numVol 5 -overwrite=yes -cleanup=no\n";
  print "    perl vsnapperf.pl seqwrite -fileSize 50g -numVol 5 -iterations 10 -random medium -overwrite=yes\n";
  print "    perl vsnapperf.pl seqwrite -fileSize 50g -numVol 5 -iterations 20 -random medium -skipNum 10\n";
  print "    perl vsnapperf.pl seqread -fileSize 50g -numVol 5 -iterations 4\n";
  print "    perl vsnapperf.pl cleanup -numVol 5 -volName mySPPtest\n";
  exit;
}

